using System;
using System.Windows.Forms;

using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL.VAL;

namespace gov.va.med.vbecs.GUI
{
	/// <summary>
	/// Layering component providing late binding for VistALink and UI components. 
	/// Created to decouple VistALink data access classes from UI implementation.
	/// </summary>
	public class VistALinkUIBinder : IVistALinkUIBinder
	{
		private Form _parentWindow;

		/// <summary>
		/// Constructs binder object.
		/// </summary>
		/// <param name="parentWindow">Reference to parent window for modal dialogs invoked by VistALink.</param>
		public VistALinkUIBinder( Form parentWindow ) 
		{
			if( parentWindow == null )
				throw( new ArgumentNullException( "parentWindow" ) );
			
			_parentWindow = parentWindow;
		}

		/// <summary>
		/// Method getting VistA access and verify codes from user via VistA sign-on dialog.
		/// </summary>
		/// <param name="serverInfo">Remote server info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="serverConnectionInfo">Remote server connection info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="accessCode">Out parameter used to return access code entered by user.</param>
		/// <param name="verifyCode">Out parameter used to return verify code entered by user.</param>
		/// <param name="requestVerifyCodeChangeFlag">Out parameter indicating user intent to change verify code.</param>
		/// <returns>True if user enters valid data and confirms the selection. False if user elects to cancel VistA sign-on.</returns>
		public virtual bool GetVistAAccessAndVerifyCodes( RemoteServerSetupAndIntroInfo serverInfo, ServerConnectionInfo serverConnectionInfo, out VistASecurityCode accessCode, out VistASecurityCode verifyCode, out bool requestVerifyCodeChangeFlag )
		{
			if( serverInfo == null )
				throw( new ArgumentNullException( "serverInfo" ) );

			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

			accessCode = verifyCode = null;
			requestVerifyCodeChangeFlag = false;

			DlgVistASignOn _dlg = null;
            DialogResult d_res = DialogResult.None;
            //dkozlov
            _parentWindow.Invoke(new MethodInvoker(delegate {
                _dlg = new DlgVistASignOn(serverInfo, serverConnectionInfo);
                d_res = _dlg.ShowDialog(_parentWindow); 
            }));

            //if (_dlg.ShowDialog(_parentWindow) != DialogResult.OK)
            //    return ReturnResultWithAppWindowsRefresh(false);
            if (d_res != DialogResult.OK)
                return ReturnResultWithAppWindowsRefresh( false );
            if (_dlg == null)
                throw (new ArgumentNullException("accessCode"));
		    accessCode = (_dlg != null) ? _dlg.GetAccessCode() : accessCode = verifyCode = null;
			verifyCode = (_dlg != null) ? _dlg.GetVerifyCode() : verifyCode = accessCode = null;
		    requestVerifyCodeChangeFlag = (_dlg != null) ? _dlg.IsVerifyCodeChangeRequired : false;
            return ReturnResultWithAppWindowsRefresh( true );				
		}

		/// <summary>
		/// Method invoked in case if user wants to change VistA verify code during sign-on.
		/// Gets old, new and new check codes from the user (new check is the same as new verify code, required for confirmation).
		/// </summary>
		/// <param name="serverInfo">Remote server info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="serverConnectionInfo">Remote server connection info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="oldVerifyCode">Out parameter used to return old verify code entered by user.</param>
		/// <param name="newVerifyCode">Out parameter used to return new verify code entered by user.</param>
		/// <param name="newCheckVerifyCode">Out parameter used to return new verify code check entered by user.</param>
		/// <returns>True if user enters valid data and confirms the selection. False if user elects to cancel VistA sign-on.</returns>
		public virtual bool GetVistANewVerifyCode( RemoteServerSetupAndIntroInfo serverInfo, ServerConnectionInfo serverConnectionInfo, out VistASecurityCode oldVerifyCode, out VistASecurityCode newVerifyCode, out VistASecurityCode newCheckVerifyCode )
		{
			if( serverInfo == null )
				throw( new ArgumentNullException( "serverInfo" ) );

			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

			oldVerifyCode = newVerifyCode = newCheckVerifyCode = null;

			DlgVistAChangeVerifyCode _dlg = new DlgVistAChangeVerifyCode( serverInfo, serverConnectionInfo );

			if( _dlg.ShowDialog( _parentWindow ) != DialogResult.OK )
				return ReturnResultWithAppWindowsRefresh( false );
		
			oldVerifyCode = _dlg.OldVerifyCode;
			newVerifyCode = _dlg.NewVerifyCode;
			newCheckVerifyCode = _dlg.NewCheckVerifyCode;			 
		
			return ReturnResultWithAppWindowsRefresh( true );
		}
		
		/// <summary>
		/// Selects first available division from list of divisions supplied by VistA. This behavior is normal
		/// since VistA division selection does not matter for most of operations performed via VsitALink. 
		/// </summary>
		/// <param name="serverInfo">Remote server info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="serverConnectionInfo">Remote server connection info provided by the <see cref="VistALink"/> internals.</param>
		/// <param name="divisionList">List of VistA divisions available for logged on user.</param>
		/// <param name="userSelectedDivision">VistA division corresponding to user-selected VBECS division.</param>
		/// <returns>
		///		Always returns true.
		///	</returns>
		public virtual bool GetVistADivisionSelection( RemoteServerSetupAndIntroInfo serverInfo, ServerConnectionInfo serverConnectionInfo, DivisionInfoCollection divisionList, out DivisionInfo userSelectedDivision )
		{
			if(  divisionList == null )
				throw( new ArgumentNullException( "divisionList" ) );

			userSelectedDivision = divisionList[0];
			return true;
		}

		/// <summary>
		/// Notifies user about the problem occured during VistA sign-on process (normally, a problem reported by remote server) 
		/// by displaying appropriate message box.
		/// </summary>
		/// <param name="message">Human-readable error messsage to display.</param>
		public virtual void NotifyAboutProblem( string message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );

			MessageBox.Show( _parentWindow, message, StrRes.OtherMsg.UC018.VistALinkSignOnErrorMessageHeader().ResString, MessageBoxButtons.OK, MessageBoxIcon.Error, MessageBoxDefaultButton.Button1 );

			_parentWindow.Refresh();
		}

		/// <summary>
		/// Checks if VistALink is available sending a heartbeat request. Attempts to reconnect if no connection is available. 
		/// Part of multiclass implementation of BR_103.01 and BR_103.02.
		/// </summary>
		/// <returns>True if VistALink is in connected state after the method call. Otherwise (VistALink disconnected) - false.</returns>
		public virtual bool CheckVistALinkReconnectIfNeeded()
		{
			if( VistALink.CheckStatus() )
				return true;

			Form _initialParentWindow = _parentWindow;

			try
			{
				DlgRestoreVistALinkConnection _dlg = new DlgRestoreVistALinkConnection();
				_parentWindow = _dlg;
				_dlg.ShowDialog( _initialParentWindow );

			    _initialParentWindow.Refresh();

			    return _dlg.IsRestoreAttemptSuccessful;
			}
			finally
			{
				_parentWindow = _initialParentWindow;
			}
		}

		/// <summary>
		/// Returns provided boolean value refreshing parent and active windows.
		/// </summary>
		/// <param name="result">Boolean value to return.</param>
		/// <returns>Supplied boolean value.</returns>
		private bool ReturnResultWithAppWindowsRefresh( bool result )
		{
            //dkozlov
            _parentWindow.Invoke(new MethodInvoker(delegate { _parentWindow.Refresh(); }));

			if( Form.ActiveForm != null && Form.ActiveForm != _parentWindow )
				Form.ActiveForm.Refresh();

			return result;
		}
	}
}
